package easik.sketch.util.ini;

import java.awt.Color;
import java.io.File;
import java.util.LinkedList;

import easik.Easik;


/**
 * A controller class that stores the values store in the ini file
 * 
 * @author Kevin Green 2006
 * @since 2006-07-11 Kevin Green
 * @version 2006-08-04 Kevin Green
 */
public class IniSettings {

	/**
	 * The ini file associated with this sketch
	 */
	private IniFile _iFile;
	/**
	 * The name of the ini file
	 */
	private final String FILENAME = "easik.ini";
	
	/**
	 * Default constructor
	 */
	public IniSettings(){
		_iFile = new IniFile();
		if(_iFile.readIni(FILENAME)){
			readUIColorSettings();
			readRecentFiles();
			readBasicSettings();
		}
		else{
			//Defaults will be used
			rewriteUIColorSettings();
			rewriteRecentFiles();
			rewriteBasicSettings();
		}
	}
	
	/**
	 * Calls for the initialization file to be writte to disk
	 * 
	 * @return True if it suceed in writing, false otherwise
	 */
	public boolean writeToFile(){
		
		//Write temp data to IniFile
		writeToIniRecentFiles();
		writeToIniColorSettings();
		writeToIniBasicSettings();
		
		//Write ini file to disk
		return _iFile.writeIni(FILENAME);
	}
	
//	----------------------------------------------------
//	-----------------------UI Colors--------------------
//	----------------------------------------------------	
	/**
	 * The color of the border of entities
	 */
	private Color ENTITY_BORDER_COLOR = 			Color.decode("#000000");
	/**
	 * The color of the background of entities
	 */
	private Color ENTITY_BG_COLOR = 				Color.decode("#FFFFFF");
	/**
	 * The color of the foreground of entities
	 */
	private Color ENTITY_FG_COLOR = 				Color.decode("#000000");
	/**
	 * The color of constraint borders
	 */
	private Color CONSTRAINT_BORDER_COLOR = 		Color.decode("#000000");
	/**
	 * The color of the background of constraints
	 */
	private Color CONSTRAINT_BG_COLOR = 			Color.decode("#EEEEEE");
	/**
	 * The color of the foreground of constraints
	 */
	private Color CONSTRAINT_FG_COLOR = 			Color.decode("#000000");
	/**
	 * The color of a simple edge
	 */
	private Color SIMPLE_EDGE_COLOR = 				Color.decode("#000000");
	/**
	 * The color of a fletched edge
	 */
	private Color FLETCHED_EDGE_COLOR = 			Color.decode("#000000");
	/**
	 * The color of a virtual edge
	 */
	private Color VIRTUAL_EDGE_COLOR = 				Color.decode("#CCCCCC");
	/**
	 * The color of a virtual highlighted edge
	 */
	private Color VIRTUAL_HIGHLIGHTED_EDGE_COLOR=	Color.decode("#444444");
	/**
	 * The path selection color
	 */
	private Color PATH_SELECTION_COLOR = 			Color.decode("#FF0000");
	/**
	 * The standard selection color
	 */
	private Color STANDARD_SELECTION_COLOR = 		Color.decode("#0000FF");

	/**
	 * @return Returns the cONSTRAINT_BG_COLOR.
	 */
	public Color getCONSTRAINT_BG_COLOR() {
		return CONSTRAINT_BG_COLOR;
	}

	/**
	 * @param constraint_bg_color The cONSTRAINT_BG_COLOR to set.
	 */
	public void setCONSTRAINT_BG_COLOR(Color constraint_bg_color) {
		CONSTRAINT_BG_COLOR = constraint_bg_color;
	}

	/**
	 * @return Returns the cONSTRAINT_BORDER_COLOR.
	 */
	public Color getCONSTRAINT_BORDER_COLOR() {
		return CONSTRAINT_BORDER_COLOR;
	}

	/**
	 * @param constraint_border_color The cONSTRAINT_BORDER_COLOR to set.
	 */
	public void setCONSTRAINT_BORDER_COLOR(Color constraint_border_color) {
		CONSTRAINT_BORDER_COLOR = constraint_border_color;
	}

	/**
	 * @return Returns the cONSTRAINT_FG_COLOR.
	 */
	public Color getCONSTRAINT_FG_COLOR() {
		return CONSTRAINT_FG_COLOR;
	}

	/**
	 * @param constraint_fg_color The cONSTRAINT_FG_COLOR to set.
	 */
	public void setCONSTRAINT_FG_COLOR(Color constraint_fg_color) {
		CONSTRAINT_FG_COLOR = constraint_fg_color;
	}

	/**
	 * @return Returns the eNTITY_BG_COLOR.
	 */
	public Color getENTITY_BG_COLOR() {
		return ENTITY_BG_COLOR;
	}

	/**
	 * @param entity_bg_color The eNTITY_BG_COLOR to set.
	 */
	public void setENTITY_BG_COLOR(Color entity_bg_color) {
		ENTITY_BG_COLOR = entity_bg_color;
	}

	/**
	 * @return Returns the eNTITY_BORDER_COLOR.
	 */
	public Color getENTITY_BORDER_COLOR() {
		return ENTITY_BORDER_COLOR;
	}

	/**
	 * @param entity_border_color The eNTITY_BORDER_COLOR to set.
	 */
	public void setENTITY_BORDER_COLOR(Color entity_border_color) {
		ENTITY_BORDER_COLOR = entity_border_color;
	}

	/**
	 * @return Returns the eNTITY_FG_COLOR.
	 */
	public Color getENTITY_FG_COLOR() {
		return ENTITY_FG_COLOR;
	}

	/**
	 * @param entity_fg_color The eNTITY_FG_COLOR to set.
	 */
	public void setENTITY_FG_COLOR(Color entity_fg_color) {
		ENTITY_FG_COLOR = entity_fg_color;
	}

	/**
	 * @return Returns the fLETCHED_EDGE_COLOR.
	 */
	public Color getFLETCHED_EDGE_COLOR() {
		return FLETCHED_EDGE_COLOR;
	}

	/**
	 * @param fletched_edge_color The fLETCHED_EDGE_COLOR to set.
	 */
	public void setFLETCHED_EDGE_COLOR(Color fletched_edge_color) {
		FLETCHED_EDGE_COLOR = fletched_edge_color;
	}

	/**
	 * @return Returns the pATh_SELECTION_COLOR.
	 */
	public Color getPATH_SELECTION_COLOR() {
		return PATH_SELECTION_COLOR;
	}

	/**
	 * @param selection_color The pATH_SELECTION_COLOR to set.
	 */
	public void setPATH_SELECTION_COLOR(Color selection_color) {
		PATH_SELECTION_COLOR = selection_color;
	}

	/**
	 * @return Returns the sTANDARD_SELECTION_COLOR.
	 */
	public Color getSTANDARD_SELECTION_COLOR() {
		return STANDARD_SELECTION_COLOR;
	}

	/**
	 * @param standard_selection_color The sTANDARD_SELECTION_COLOR to set.
	 */
	public void setSTANDARD_SELECTION_COLOR(Color standard_selection_color) {
		STANDARD_SELECTION_COLOR = standard_selection_color;
	}

	/**
	 * @return Returns the sIMPLE_EDGE_COLOR.
	 */
	public Color getSIMPLE_EDGE_COLOR() {
		return SIMPLE_EDGE_COLOR;
	}

	/**
	 * @param simple_edge_color The sIMPLE_EDGE_COLOR to set.
	 */
	public void setSIMPLE_EDGE_COLOR(Color simple_edge_color) {
		SIMPLE_EDGE_COLOR = simple_edge_color;
	}

	/**
	 * @return Returns the vIRTUAL_EDGE_COLOR.
	 */
	public Color getVIRTUAL_EDGE_COLOR() {
		return VIRTUAL_EDGE_COLOR;
	}

	/**
	 * @param virtual_edge_color The vIRTUAL_EDGE_COLOR to set.
	 */
	public void setVIRTUAL_EDGE_COLOR(Color virtual_edge_color) {
		VIRTUAL_EDGE_COLOR = virtual_edge_color;
	}

	/**
	 * @return Returns the vIRTUAL_HIGHLIGHTED_EDGE_COLOR.
	 */
	public Color getVIRTUAL_HIGHLIGHTED_EDGE_COLOR() {
		return VIRTUAL_HIGHLIGHTED_EDGE_COLOR;
	}

	/**
	 * @param virtual_highlighted_edge_color The vIRTUAL_HIGHLIGHTED_EDGE_COLOR to set.
	 */
	public void setVIRTUAL_HIGHLIGHTED_EDGE_COLOR(
			Color virtual_highlighted_edge_color) {
		VIRTUAL_HIGHLIGHTED_EDGE_COLOR = virtual_highlighted_edge_color;
	}
	
	/**
	 * Reads the UI Color Settings in from the initialization file
	 */
	public void readUIColorSettings(){
		if(_iFile.getHeader("UI Color Settings") == null)
			rewriteUIColorSettings();
		
		IniFileNode node = _iFile.getHeader("UI Color Settings");
		
		//Read ENTITY_BORDER_COLOR
		try{
			String color = node.getItem("ENTITY_BORDER_COLOR");
			if(color != null)
				ENTITY_BORDER_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read ENTITY_BG_COLOR
		try{
			String color = node.getItem("ENTITY_BG_COLOR");
			if(color != null)
				ENTITY_BG_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read ENTITY_FG_COLOR
		try{
			String color = node.getItem("ENTITY_FG_COLOR");
			if(color != null)
				ENTITY_FG_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read CONSTRAINT_BORDER_COLOR
		try{
			String color = node.getItem("CONSTRAINT_BORDER_COLOR");
			if(color != null)
				CONSTRAINT_BORDER_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read CONSTRAINT_BG_COLOR
		try{
			String color = node.getItem("CONSTRAINT_BG_COLOR");
			if(color != null)
				CONSTRAINT_BG_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read CONSTRAINT_FG_COLOR
		try{
			String color = node.getItem("CONSTRAINT_FG_COLOR");
			if(color != null)
				CONSTRAINT_FG_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read SIMPLE_EDGE_COLOR
		try{
			String color = node.getItem("SIMPLE_EDGE_COLOR");
			if(color != null)
				SIMPLE_EDGE_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read FLETCHED_EDGE_COLOR
		try{
			String color = node.getItem("FLETCHED_EDGE_COLOR");
			if(color != null)
				FLETCHED_EDGE_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read VIRTUAL_EDGE_COLOR
		try{
			String color = node.getItem("VIRTUAL_EDGE_COLOR");
			if(color != null)
				VIRTUAL_EDGE_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read VIRTUAL_HIGHLIGHTED_EDGE_COLOR
		try{
			String color = node.getItem("VIRTUAL_HIGHLIGHTED_EDGE_COLOR");
			if(color != null)
				VIRTUAL_HIGHLIGHTED_EDGE_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read PATH_SELECTION_COLOR
		try{
			String color = node.getItem("PATH_SELECTION_COLOR");
			if(color != null)
				PATH_SELECTION_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read STANDARD_SELECTION_COLOR
		try{
			String color = node.getItem("STANDARD_SELECTION_COLOR");
			if(color != null)
				STANDARD_SELECTION_COLOR = Color.decode(color);
		}
		catch(Exception e){
			//Could not be read
		}
		
	}
	
	/**
	 * Returns the standard web code for the given color
	 * 
	 * @param inColor The color to be converted
	 * @return The standard web color representation
	 */
	public String colorToString(Color inColor){
		String color = Integer.toHexString(inColor.getRGB() & 16777215);
		while(color.length() < 6){
			color = "0" + color;
		}
		color = color.toUpperCase();
		color = "#" + color;
		return color;
	}
	
	/**
	 * Rewrites the UI Color Settings to the initialization file
	 */
	public void rewriteUIColorSettings(){
		_iFile.addHeader("UI Color Settings");
		IniFileNode node = _iFile.getHeader("UI Color Settings");
		node.addItem("ENTITY_BORDER_COLOR", colorToString(ENTITY_BORDER_COLOR));
		node.addItem("ENTITY_BG_COLOR", colorToString(ENTITY_BG_COLOR));
		node.addItem("ENTITY_FG_COLOR", colorToString(ENTITY_FG_COLOR));
		node.addItem("CONSTRAINT_BORDER_COLOR", colorToString(CONSTRAINT_BORDER_COLOR));
		node.addItem("CONSTRAINT_BG_COLOR", colorToString(CONSTRAINT_BG_COLOR));
		node.addItem("CONSTRAINT_FG_COLOR", colorToString(CONSTRAINT_FG_COLOR));
		node.addItem("SIMPLE_EDGE_COLOR", colorToString(SIMPLE_EDGE_COLOR));
		node.addItem("FLETCHED_EDGE_COLOR", colorToString(FLETCHED_EDGE_COLOR));
		node.addItem("VIRTUAL_EDGE_COLOR", colorToString(VIRTUAL_EDGE_COLOR));
		node.addItem("VIRTUAL_HIGHLIGHTED_EDGE_COLOR", colorToString(VIRTUAL_HIGHLIGHTED_EDGE_COLOR));
		node.addItem("PATH_SELECTION_COLOR", colorToString(PATH_SELECTION_COLOR));
		node.addItem("STANDARD_SELECTION_COLOR", colorToString(STANDARD_SELECTION_COLOR));
	}
	
	/**
	 * Writes temporary UI Color Settings information to the IniFile
	 */
	public void writeToIniColorSettings(){
		if(_iFile.getHeader("UI Color Settings") == null)
			rewriteUIColorSettings();
		else{
			IniFileNode node = _iFile.getHeader("UI Color Settings");
			node.addItem("ENTITY_BORDER_COLOR", colorToString(ENTITY_BORDER_COLOR));
			node.addItem("ENTITY_BG_COLOR", colorToString(ENTITY_BG_COLOR));
			node.addItem("ENTITY_FG_COLOR", colorToString(ENTITY_FG_COLOR));
			node.addItem("CONSTRAINT_BORDER_COLOR", colorToString(CONSTRAINT_BORDER_COLOR));
			node.addItem("CONSTRAINT_BG_COLOR", colorToString(CONSTRAINT_BG_COLOR));
			node.addItem("CONSTRAINT_FG_COLOR", colorToString(CONSTRAINT_FG_COLOR));
			node.addItem("SIMPLE_EDGE_COLOR", colorToString(SIMPLE_EDGE_COLOR));
			node.addItem("FLETCHED_EDGE_COLOR", colorToString(FLETCHED_EDGE_COLOR));
			node.addItem("VIRTUAL_EDGE_COLOR", colorToString(VIRTUAL_EDGE_COLOR));
			node.addItem("VIRTUAL_HIGHLIGHTED_EDGE_COLOR", colorToString(VIRTUAL_HIGHLIGHTED_EDGE_COLOR));
			node.addItem("PATH_SELECTION_COLOR", colorToString(PATH_SELECTION_COLOR));
			node.addItem("STANDARD_SELECTION_COLOR", colorToString(STANDARD_SELECTION_COLOR));
		}
	}
	
//	-------------------------------------------------------------
//	----------------------Basic Settings-------------------------
//	-------------------------------------------------------------
	/**
	 * Last used folder
	 */
	private String lastUsedFolder = System.getProperty("user.dir");
	/**
	 * Specific Folder
	 */
	private String specificFolder = System.getProperty("user.dir");
	/**
	 * Option to use the last used folder as the default folder
	 */
	public static final int FOLDER_LAST_USED = 1;
	/**
	 * Option to use the program running directory as the default folder
	 */
	public static final int FOLDER_RUNNING = 2;
	/**
	 * Option to use the specified
	 */
	public static final int FOLDER_SPECIFIC = 3;
	/**
	 * Default save/open folder operation
	 */
	private int saveOpenOperation = FOLDER_LAST_USED;
	/**
	 * The default value for show all attributes & unique keys option when app is loaded
	 */
	private boolean showAllAttsAndUniqueKeys = true;
	
	/**
	 * @return Returns the lastUsedFolder.
	 */
	public String getLastUsedFolder() {
		return lastUsedFolder;
	}

	/**
	 * @param lastUsedFolder The lastUsedFolder to set.
	 */
	public void setLastUsedFolder(String lastUsedFolder) {
		this.lastUsedFolder = lastUsedFolder;
	}

	/**
	 * @return Returns the saveOpenOperation.
	 */
	public int getSaveOpenOperation() {
		return saveOpenOperation;
	}

	/**
	 * @param saveOpenOperation The saveOpenOperation to set.
	 */
	public void setSaveOpenOperation(int saveOpenOperation) {
		this.saveOpenOperation = saveOpenOperation;
	}

	/**
	 * @return Returns the showAllAttsAndUniqueKeys.
	 */
	public boolean isShowAllAttsAndUniqueKeys() {
		return showAllAttsAndUniqueKeys;
	}

	/**
	 * @param showAllAttsAndUniqueKeys The showAllAttsAndUniqueKeys to set.
	 */
	public void setShowAllAttsAndUniqueKeys(boolean showAllAttsAndUniqueKeys) {
		this.showAllAttsAndUniqueKeys = showAllAttsAndUniqueKeys;
	}

	/**
	 * @return Returns the specificFolder.
	 */
	public String getSpecificFolder() {
		return specificFolder;
	}

	/**
	 * @param specificFolder The specificFolder to set.
	 */
	public void setSpecificFolder(String specificFolder) {
		this.specificFolder = specificFolder;
	}
	
	/**
	 * Returns the default folder to be used for open and save
	 * operations based on the current settings
	 * 
	 * @return The default folder to be used for open and save operations
	 */
	public String getDefaultFolder(){
		if(saveOpenOperation == FOLDER_SPECIFIC){
			return specificFolder;
		}
		else if(saveOpenOperation == FOLDER_RUNNING){
			return System.getProperty("user.dir");
		}
		else{
			return lastUsedFolder;
		}
	}
	
	/**
	 * Reads the recent files from the initialization file
	 */
	public void readBasicSettings(){
		if(_iFile.getHeader("Basic Settings") == null)
			rewriteBasicSettings();
		
		IniFileNode node = _iFile.getHeader("Basic Settings");
		
		//Read default folder operation
		try{
				String op = node.getItem("Save And Open Operation");
				if(op.equals("Last Used")){
					saveOpenOperation = FOLDER_LAST_USED;
				}
				else if(op.equals("Running Directory")){
					saveOpenOperation = FOLDER_RUNNING;
				}
				else if(op.equals("Specific")){
					saveOpenOperation = FOLDER_SPECIFIC;
				}
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read the last used and specific folder
		try{
			String op = node.getItem("Last Used Folder");
			if(op != null && !op.equals("")){
				lastUsedFolder = op;
			}
			String op2 = node.getItem("Specific Folder");
			if(op2 != null && !op2.equals("")){
				specificFolder = op2;
			}
		}
		catch(Exception e){
			//Could not be read
		}
		
		//Read the default value for show all attributes & unique keys option when app is loaded
		try{
			String op = node.getItem("Show Attributes And Unique Keys");
			showAllAttsAndUniqueKeys = Boolean.parseBoolean(op);
		}
		catch(Exception e){
			//Could not be read
		}
	}
	
	/**
	 * Initializes the recent files list
	 */
	public void rewriteBasicSettings(){
		_iFile.addHeader("Basic Settings");
		IniFileNode node = _iFile.getHeader("Basic Settings");
		
		String op = "";
		if(saveOpenOperation == FOLDER_SPECIFIC){
			op = "Specific";
		}
		else if(saveOpenOperation == FOLDER_RUNNING){
			op = "Running Directory";
		}
		else{
			op = "Last Used";
		}
		node.addItem("Save And Open Operation", op);
		node.addItem("Last Used Folder", lastUsedFolder);
		node.addItem("Specific Folder", specificFolder);
		node.addItem("Show Attributes And Unique Keys", String.valueOf(showAllAttsAndUniqueKeys));
	}
	
	/**
	 * Writes the recent files to the IniFile
	 */
	public void writeToIniBasicSettings(){
		if(_iFile.getHeader("Basic Settings") == null)
			rewriteRecentFiles();
		
		IniFileNode node = _iFile.getHeader("Basic Settings");
		
		String op = "";
		if(saveOpenOperation == FOLDER_SPECIFIC){
			op = "Specific";
		}
		else if(saveOpenOperation == FOLDER_RUNNING){
			op = "Running Directory";
		}
		else{
			op = "Last Used";
		}
		node.addItem("Save And Open Operation", op);
		node.addItem("Last Used Folder", lastUsedFolder);
		node.addItem("Specific Folder", specificFolder);
		node.addItem("Show Attributes And Unique Keys", String.valueOf(showAllAttsAndUniqueKeys));
	}
	
	
//	-------------------------------------------------------------
//	-----------------------Recent Files--------------------------
//	-------------------------------------------------------------
	/**
	 * The list of the most recent files used
	 */
	private LinkedList<String> RECENT_FILES = new LinkedList<String>();
	/**
	 * The number of most recent files, constrained to eight
	 */
	private int numRecentFiles = 8;

	/**
	 * @return Returns the rECENT_FILES.
	 */
	public LinkedList<String> getRECENT_FILES() {
		return RECENT_FILES;
	}

	/**
	 * @param recent_files The rECENT_FILES to set.
	 */
	public void setRECENT_FILES(LinkedList<String> recent_files) {
		RECENT_FILES = recent_files;
	}
	
	/**
	 * Adds a file to the top of the recent file list
	 * 
	 * @param inFile The file to be added
	 */
	public void addRecentFile(File inFile){
		String path = inFile.getPath();
		
		//Check if it is already in the recent files
		if(RECENT_FILES.contains(path)){
			RECENT_FILES.remove(path);
			RECENT_FILES.addFirst(path);
		}
		else{
			RECENT_FILES.addFirst(path);
		}
		
		//Check to see if there are too many files
		while(RECENT_FILES.size() > numRecentFiles){
			RECENT_FILES.removeLast();
		}
		
		//Reload menu items
		Easik.getInstance().getFrame().updateRecentFilesMenu();
	}
	
	/**
	 * Reads the recent files from the initialization file
	 */
	public void readRecentFiles(){
		if(_iFile.getHeader("Recent Files") == null)
			rewriteRecentFiles();
		
		IniFileNode node = _iFile.getHeader("Recent Files");
		
		//Read recent files
		try{
			for(int i=0; i<numRecentFiles; i++){
				String recent = node.getItem("RECENT"+i);
				if(recent != null && !recent.equals("")){
					RECENT_FILES.addLast(recent);
				}
			}
		}
		catch(Exception e){
			//Could not be read
		}
	}
	
	/**
	 * Initializes the recent files list
	 */
	public void rewriteRecentFiles(){
		_iFile.addHeader("Recent Files");
		IniFileNode node = _iFile.getHeader("Recent Files");
		for(int i=0; i<numRecentFiles; i++){
			node.addItem("RECENT"+i, "");
		}
	}
	
	/**
	 * Writes the recent files to the IniFile
	 */
	public void writeToIniRecentFiles(){
		if(_iFile.getHeader("Recent Files") == null)
			rewriteRecentFiles();
		
		IniFileNode node = _iFile.getHeader("Recent Files");
		
		for(int i=0; i<RECENT_FILES.size(); i++){
			node.addItem("RECENT"+i, (String)RECENT_FILES.get(i));
		}
	}
}
